/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
    with Sandia Corporation, the U.S. Government retains certain 
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov      
   
  ***************************************************************** */
// -*- Mode : c++; tab-width: 3; c-tab-always-indent: t; indent-tabs-mode: nil; c-basic-offset: 3 -*-

/*! \file IntPtTemplate.hpp

Header file for the Mesquite::LInfTemplate class

  \author Shankar Prasad Sastry
  \date   9th July 2008
 */


#ifndef IntPtTemplate_hpp
#define IntPtTemplate_hpp

#include "Mesquite.hpp"
#include "ObjectiveFunctionTemplate.hpp"
#include "QualityMetric.hpp"
#include "MsqError.hpp"


namespace MESQUITE_NS
{
  /*! \class IntPtTemplate
    \brief Computes the Interior Point Method's objective function for a given patch,
    i.e., IntPtTemplate::concrete_evaluate returns the maximum absolute value of
    the quality metric values  on 'patch' and sums up the barrier functions
  */
   class IntPtTemplate:public ObjectiveFunctionTemplate
   {
   public:
     IntPtTemplate(QualityMetric *qualitymetric):ObjectiveFunctionTemplate(qualitymetric) 
     {
 	mu=0.2, mufac=0.9; t=-1e30;
     }

     virtual ~IntPtTemplate(){}
     virtual bool evaluate( EvalType type, 
                            PatchData& pd,
                            double& value_out,
                            bool free,
                            MsqError& err )

{
  /*if (type != ObjectiveFunction::CALCULATE) {
    MSQ_SETERR(err)(
      "IntPtTemplate does not support block coodinate descent algoritms",
      MsqError::INVALID_STATE );
    return false;
  }*/

  QualityMetric* qm = get_quality_metric();
  qm->get_evaluations( pd, qmHandles, free, err );  MSQ_ERRFALSE(err);
  const double negate = qm->get_negate_flag();
  
  // calculate OF value for just the patch
  msq_std::vector<size_t>::const_iterator i;
  double value, log_sum=0.0;
  qm->tt = t;
  //printf("mu= %lf t=%lf log_sum = %lf Value out = %lf\n", mu, t, log_sum, value_out); 
  for (i = qmHandles.begin(); i != qmHandles.end(); ++i)
  {
    value = 1e6;
    bool result = qm->evaluate( pd, *i, value, err);
    if (MSQ_CHKERR(err))
      return false;
    if (!result)
	{
	//printf("intpt here\n");
      return false;
	}

    log_sum += value;
    //printf("\tvalue = %lf, min_t = %lf, log= %lf logsum = %lf\n", value, t, log(value-t), log_sum);
  }
  value_out = -t - mu*log_sum;
  //printf("mu= %lf t=%lf log_sum = %lf Value out = %lf\n", mu, t, log_sum, value_out); 
  return true;
}



     virtual ObjectiveFunction* clone() const
	{ return new IntPtTemplate(get_quality_metric()); }
     virtual void clear(){}
     double mu, t, mufac;
   private:
     /** Temporary storage for qm sample handles */
     mutable msq_std::vector<size_t> qmHandles;
   };
}//namespace
#endif // IntPtTemplate_hpp

